// ==UserScript==
// @name         5ch URLプレビューカード表示（簡易OGP）
// @namespace    http://tampermonkey.net/
// @version      0.6
// @description  OGP取得失敗時に1ページ目へ再試行。レイアウト崩れも改善。TikTok CDN動画除外追加。
// @match        *://*.5ch.net/test/read.cgi/*
// @grant        GM_xmlhttpRequest
// @connect      *
// ==/UserScript==

(function () {
  'use strict';

  const processedLinks = new Set();

  function shouldExcludeLink(href) {
    try {
      const urlObj = new URL(href);
      const hostname = urlObj.hostname;
      const pathname = urlObj.pathname;

      const imagePattern = /\.(jpg|jpeg|png|gif|bmp|svg)(\?.*|#.*)?$/i;
      const videoFilePattern = /\.(mp4|webm|mov|avi|flv|mkv)(\?.*|#.*)?$/i;
      const videoSitePattern = /youtube\.com|youtu\.be|vimeo\.com/i;
      const twitterPattern = /^(.*\.)?(twitter\.com|x\.com|twimg\.com)$/i;
      const fivechPattern = /5ch\.net/i;

      // TikTok CDNドメインかつパスに /video/ を含む場合は除外
      if (hostname.endsWith("tiktokcdn.com") && pathname.includes("/video/")) {
        return true;
      }

      return imagePattern.test(href)
        || videoFilePattern.test(href)
        || videoSitePattern.test(hostname)
        || twitterPattern.test(hostname)
        || fivechPattern.test(hostname);
    } catch {
      return false;
    }
  }

  function normalizeURL(href) {
    return href.replace("//www.asahi.com/sp/", "//www.asahi.com/");
  }

  function toAbsoluteURL(base, relative) {
    try {
      return new URL(relative, base).href;
    } catch {
      return relative;
    }
  }

  function getFirstPageURLIfNeeded(url) {
    try {
      const u = new URL(url);
      const match = u.pathname.match(/^(.*?\/\d+)\/\d+\/?$/);
      if (match) {
        u.pathname = match[1] + '/';
        return u.toString();
      }
    } catch {}
    return null;
  }

  function createPreviewCard(ogData, href) {
    const container = document.createElement("div");
    const shadow = container.attachShadow({ mode: "open" });

    const style = document.createElement("style");
    style.textContent = `
      .card {
        margin: 10px 0;
        padding: 10px;
        border: 1px solid #ccc;
        border-radius: 8px;
        background-color: #303030;
        color: #fff;
        font-family: sans-serif;
        max-width: 600px;
        display: flex;
        flex-direction: row;
        gap: 10px;
        align-items: flex-start;
      }
      .card a {
        text-decoration: none;
        color: inherit;
        display: flex;
        flex-direction: row;
        gap: 10px;
        align-items: flex-start;
        width: 100%;
      }
      .card img {
        width: 80px;
        height: 80px;
        object-fit: cover;
        display: block;
        flex-shrink: 0;
        border-radius: 4px;
      }
      .content {
        display: flex;
        flex-direction: column;
        justify-content: center;
        flex: 1;
      }
      .title {
        font-weight: bold;
        margin-bottom: 4px;
        line-height: 1.3;
      }
      .domain {
        font-size: 0.8em;
        opacity: 0.7;
        text-align: right;
      }
    `;

    const card = document.createElement("div");
    card.className = "card";

    const link = document.createElement("a");
    link.href = href;
    link.target = "_blank";

    if (ogData.image) {
      const img = document.createElement("img");
      img.src = ogData.image;
      link.appendChild(img);
    }

    const content = document.createElement("div");
    content.className = "content";

    const title = document.createElement("div");
    title.className = "title";
    title.textContent = ogData.title || "（タイトルなし）";

    const domain = document.createElement("div");
    domain.className = "domain";
    domain.textContent = new URL(href).hostname;

    content.appendChild(title);
    content.appendChild(domain);
    link.appendChild(content);
    card.appendChild(link);
    shadow.appendChild(style);
    shadow.appendChild(card);

    return container;
  }

  function fetchOGP(url, callback, fallbackTried = false) {
    GM_xmlhttpRequest({
      method: "GET",
      url: url,
      headers: {
        "User-Agent": "Mozilla/5.0",
        "Accept-Language": "ja,en;q=0.9",
        "Referer": "https://www.google.com"
      },
      onload: function (res) {
        const parser = new DOMParser();
        const doc = parser.parseFromString(res.responseText, "text/html");

        const getMeta = (sel) => doc.querySelector(sel)?.content;

        const ogTitle =
          getMeta('meta[property="og:title"]') ||
          getMeta('meta[name="twitter:title"]') ||
          getMeta('meta[name="title"]') ||
          doc.title;

        const rawImage =
          getMeta('meta[property="og:image"]') ||
          getMeta('meta[name="twitter:image"]');

        const ogImage = rawImage ? toAbsoluteURL(url, rawImage) : null;

        if (!ogTitle && !fallbackTried) {
          const firstPageUrl = getFirstPageURLIfNeeded(url);
          if (firstPageUrl) {
            console.log(`[OGP Retry] ${url} → ${firstPageUrl}`);
            fetchOGP(firstPageUrl, callback, true);
            return;
          }
        }

        callback({ title: ogTitle || "(取得失敗)", image: ogImage });
      },
      onerror: function () {
        callback({ title: "(取得失敗)", image: null });
      },
    });
  }

  function processLink(link) {
    const rawHref = link.href;

    // 同一URLでもリクエストしカードを作るが除外判定は適用
    if (!rawHref.startsWith("http") || shouldExcludeLink(rawHref)) return;

    const normalizedHref = normalizeURL(rawHref);

    processedLinks.add(rawHref);
    fetchOGP(normalizedHref, function (ogData) {
      const card = createPreviewCard(ogData, normalizedHref);
      link.parentElement.insertBefore(card, link.nextSibling);
    });
  }

  function scanLinks() {
    const links = document.querySelectorAll("a[href]");
    links.forEach(link => observer.observe(link));
  }

  const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
      if (entry.isIntersecting) {
        processLink(entry.target);
        observer.unobserve(entry.target);
      }
    });
  }, {
    rootMargin: "200px",
  });

  window.addEventListener("load", () => scanLinks());

})();